/*

Multichannel Simple Light Dimmer Project
Author: Mauro Grassi. Nov 2008.
For Silicon Chip.
Features, To Do List:

(a) Multi Channel 
(b) Automatic TimeOut: done! How to set?
(c) Learning Remote Control Codes, to EEPROM
(d) non-dimmable-Compact-Fluoro modes (ZV mode) and 12V Halogens?
(e) baby mode.

Pin Outs for IC1: PIC18F1320

Pin			I/O	- Function 

RA0 (1) 					O	- RGB LED
RA1 (2)						O	- RGB LED
RA2 (6)						O	- RGB LED
RA3 (7)						O	- RGB LED
RA4 (3)						O	- RGB LED	
RA5							I	- MCLR/VPP
RA6							I	- OSC1
RA7 						I	- OSC2

RB0/INT0 (8)				I	- mains sync
RB1 (9)						O 	- RGB LED
RB2 (17)					I	- connected to RB3/CCP1
RB3/CCP1 (18)				O 	- output for Triac Trigger
RB4 (10) 		KBI0		I	- data input from infra red module IRD1
RB5 (11) 		KBI1		O	- N.C.
RB6 (12) 		KBI2		O	- PGC
RB7 (13) 		KBI3		O 	- PGD

This version 1.00 is pretty much working (at least the low level drivers are working)
Version History:

Version 1.10: improved stability, added target levels and timeout period in double seconds (from the mains), added key repeat semantics,
Version 1.12: added EEPROM routines
Version 1.14: added Codes in EEPROM.
Version 1.16: improved values and order of interrupts for stability, minor changes to CCP1CON sequence. 
Also the BOR level is set to 2.7V just in case (so it doesn't reset on the transient current peaks) and the loop has count=350 for delayms with 2 nops
to shorten the LED drive time, and minimize power consumption!
  
As of version 2.00 we moved from an instruction clock of 5MHz= 20Mhz crystal /4 to HSPLL with 10MHz crystal (doubled MIPS)

Version History (Mauro Grassi)

Version 2.20 is the first version to fix unreliability of IR decoding due to excessive power consumption of light when on
			 From this version onwards, we require two identical IR codes to be received till we process the key press...

Version 2.60 adds power saving and a different dimming curve (an actual zero current throught filament when 'off'- as previous versions the lamp seemed off but there was
a small current flowing through the filament, because the off phase trigger point was not late enough, this version fixes that).

Version 2.64 adds improvement in standby power conservation by going to sleep and waking up with change on pin interrupt for the infra red module.

Version 2.90 narrows the turn on pulse for the gate for improved stability and increases IR_REPEAT to 2 so 3 consecutive codes are needed.

Version 2.92 adds persistent brightness levels (useful if you are installing a series switch for eg to minimize standby power).

Version 2.96 added better key error correction, and better halogen dimming with support for asymmetrical phases!

Version 3.80:

adds support for signed enterNumber() and blinkNumber(); offset is added for one phase and subtracted for the other.

Version 3.85: adds support for individual offset settings for each phase.

Version 4.20: almost final version

Version 4.30: this version improves on 4.20 by temporarily making the CCP1 pin an input while decoding the IR signal. This improves the IR sensitivity substantially.

*/

#include <p18f1320.h>
#include "ir.h"

#pragma config OSC      = HSPLL
#pragma config DEBUG	= OFF
#pragma config IESO     = ON
#pragma config PWRT     = ON
#pragma config FSCM		= OFF
#pragma config BOR      = OFF
#pragma config BORV		= 27
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = OFF
#pragma config LVP      = OFF
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRTB     = OFF      		// Boot Block Write Protection
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTRB    = OFF
//
// Definitions
//
typedef unsigned char byte;
//
void myISRHigh(void);
void myISR(void);
extern void initRC5(void);
void setLampBrightness(byte);
//void setLampBrightnessLevel(byte);
//void setQuiescentBrightnessLevel(byte);
extern void reDefineRemoteControlCodes(void);
extern void defineDefaultRemoteControl(void);
//
int targetBrightness;
byte targetACK;
byte badCode;
byte badTimes;
byte bootMode;
int key;
byte tickCounter;
int phaseCounter;
byte seconds;
int eternalSeconds;
byte flashState;
byte flashModulus;
byte address;
byte limiting0;
byte limiting1;
byte loopDelay;
byte keyBuffer[KEY_BUFFER_SIZE];
byte keyGetPtr;
byte keyPtr;
byte keyFull;
int offset, offset1;
byte receivedAddress;
int lampBrightness;				// internal level used for interrupt servicing the Triac system
byte mode;						// mode of operation, as follows:
								// bit	7			6			5			4			3			2			1			0
								//																				ZV mode		FLASHING=1 flashing mode
byte reqLampBrightness;			// requested level
byte reqQuiescentBrightness;	// requested quiescent level
byte setBrightness;				// actual level
int timeOutPeriod;
int timeOutCounter;

#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
	//_asm goto RM_HIGH_INTERRUPT_VECTOR _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
	//_asm goto RM_LOW_INTERRUPT_VECTOR _endasm
}



// the dimming curve is capped at 245 max brightness
rom int DimmingCurve[DIMMING_LEVELS]=
{
1.000000*FIXED_DENOMINATOR,0.960185*FIXED_DENOMINATOR,0.943657*FIXED_DENOMINATOR,0.930949*FIXED_DENOMINATOR,0.920214*FIXED_DENOMINATOR,0.910737*FIXED_DENOMINATOR,0.902153*FIXED_DENOMINATOR,0.894243*FIXED_DENOMINATOR,
0.886866*FIXED_DENOMINATOR,0.879923*FIXED_DENOMINATOR,0.873343*FIXED_DENOMINATOR,0.867072*FIXED_DENOMINATOR,0.861068*FIXED_DENOMINATOR,0.855297*FIXED_DENOMINATOR,0.849733*FIXED_DENOMINATOR,0.847496*FIXED_DENOMINATOR,
0.842392*FIXED_DENOMINATOR,0.837435*FIXED_DENOMINATOR,0.832612*FIXED_DENOMINATOR,0.827911*FIXED_DENOMINATOR,0.823324*FIXED_DENOMINATOR,0.818841*FIXED_DENOMINATOR,0.814454*FIXED_DENOMINATOR,0.810158*FIXED_DENOMINATOR,
0.805945*FIXED_DENOMINATOR,0.801810*FIXED_DENOMINATOR,0.797749*FIXED_DENOMINATOR,0.793757*FIXED_DENOMINATOR,0.789830*FIXED_DENOMINATOR,0.785965*FIXED_DENOMINATOR,0.782158*FIXED_DENOMINATOR,0.778405*FIXED_DENOMINATOR,
0.774705*FIXED_DENOMINATOR,0.771055*FIXED_DENOMINATOR,0.767451*FIXED_DENOMINATOR,0.763893*FIXED_DENOMINATOR,0.760377*FIXED_DENOMINATOR,0.756903*FIXED_DENOMINATOR,0.753467*FIXED_DENOMINATOR,0.750069*FIXED_DENOMINATOR,
0.746707*FIXED_DENOMINATOR,0.743380*FIXED_DENOMINATOR,0.740085*FIXED_DENOMINATOR,0.736823*FIXED_DENOMINATOR,0.733590*FIXED_DENOMINATOR,0.730387*FIXED_DENOMINATOR,0.727213*FIXED_DENOMINATOR,0.724065*FIXED_DENOMINATOR,
0.720944*FIXED_DENOMINATOR,0.717848*FIXED_DENOMINATOR,0.714776*FIXED_DENOMINATOR,0.711728*FIXED_DENOMINATOR,0.708703*FIXED_DENOMINATOR,0.705699*FIXED_DENOMINATOR,0.702717*FIXED_DENOMINATOR,0.699755*FIXED_DENOMINATOR,
0.696813*FIXED_DENOMINATOR,0.693891*FIXED_DENOMINATOR,0.690987*FIXED_DENOMINATOR,0.688101*FIXED_DENOMINATOR,0.685232*FIXED_DENOMINATOR,0.682381*FIXED_DENOMINATOR,0.679545*FIXED_DENOMINATOR,0.676726*FIXED_DENOMINATOR,
0.673922*FIXED_DENOMINATOR,0.671133*FIXED_DENOMINATOR,0.668358*FIXED_DENOMINATOR,0.665598*FIXED_DENOMINATOR,0.662851*FIXED_DENOMINATOR,0.660118*FIXED_DENOMINATOR,0.657397*FIXED_DENOMINATOR,0.654689*FIXED_DENOMINATOR,
0.651992*FIXED_DENOMINATOR,0.649308*FIXED_DENOMINATOR,0.646635*FIXED_DENOMINATOR,0.643973*FIXED_DENOMINATOR,0.641322*FIXED_DENOMINATOR,0.638682*FIXED_DENOMINATOR,0.636051*FIXED_DENOMINATOR,0.633431*FIXED_DENOMINATOR,
0.630820*FIXED_DENOMINATOR,0.628218*FIXED_DENOMINATOR,0.625625*FIXED_DENOMINATOR,0.623041*FIXED_DENOMINATOR,0.620466*FIXED_DENOMINATOR,0.617899*FIXED_DENOMINATOR,0.615340*FIXED_DENOMINATOR,0.612788*FIXED_DENOMINATOR,
0.610245*FIXED_DENOMINATOR,0.607708*FIXED_DENOMINATOR,0.605179*FIXED_DENOMINATOR,0.602656*FIXED_DENOMINATOR,0.600140*FIXED_DENOMINATOR,0.597631*FIXED_DENOMINATOR,0.595128*FIXED_DENOMINATOR,0.592631*FIXED_DENOMINATOR,
0.590139*FIXED_DENOMINATOR,0.587654*FIXED_DENOMINATOR,0.585174*FIXED_DENOMINATOR,0.582699*FIXED_DENOMINATOR,0.580229*FIXED_DENOMINATOR,0.577764*FIXED_DENOMINATOR,0.575304*FIXED_DENOMINATOR,0.572849*FIXED_DENOMINATOR,
0.570398*FIXED_DENOMINATOR,0.567951*FIXED_DENOMINATOR,0.565508*FIXED_DENOMINATOR,0.563069*FIXED_DENOMINATOR,0.560634*FIXED_DENOMINATOR,0.558203*FIXED_DENOMINATOR,0.555775*FIXED_DENOMINATOR,0.553350*FIXED_DENOMINATOR,
0.550928*FIXED_DENOMINATOR,0.548509*FIXED_DENOMINATOR,0.546094*FIXED_DENOMINATOR,0.543680*FIXED_DENOMINATOR,0.541270*FIXED_DENOMINATOR,0.538861*FIXED_DENOMINATOR,0.536455*FIXED_DENOMINATOR,0.534051*FIXED_DENOMINATOR,
0.531649*FIXED_DENOMINATOR,0.529249*FIXED_DENOMINATOR,0.526850*FIXED_DENOMINATOR,0.524453*FIXED_DENOMINATOR,0.522057*FIXED_DENOMINATOR,0.519663*FIXED_DENOMINATOR,0.517270*FIXED_DENOMINATOR,0.514877*FIXED_DENOMINATOR,
0.512486*FIXED_DENOMINATOR,0.510095*FIXED_DENOMINATOR,0.507705*FIXED_DENOMINATOR,0.505315*FIXED_DENOMINATOR,0.502926*FIXED_DENOMINATOR,0.500536*FIXED_DENOMINATOR,0.498147*FIXED_DENOMINATOR,0.495758*FIXED_DENOMINATOR,
0.493368*FIXED_DENOMINATOR,0.490978*FIXED_DENOMINATOR,0.488588*FIXED_DENOMINATOR,0.486196*FIXED_DENOMINATOR,0.483804*FIXED_DENOMINATOR,0.481412*FIXED_DENOMINATOR,0.479018*FIXED_DENOMINATOR,0.476623*FIXED_DENOMINATOR,
0.474226*FIXED_DENOMINATOR,0.471828*FIXED_DENOMINATOR,0.469429*FIXED_DENOMINATOR,0.467028*FIXED_DENOMINATOR,0.464624*FIXED_DENOMINATOR,0.462219*FIXED_DENOMINATOR,0.459812*FIXED_DENOMINATOR,0.457402*FIXED_DENOMINATOR,
0.454990*FIXED_DENOMINATOR,0.452576*FIXED_DENOMINATOR,0.450158*FIXED_DENOMINATOR,0.447738*FIXED_DENOMINATOR,0.445314*FIXED_DENOMINATOR,0.442888*FIXED_DENOMINATOR,0.440458*FIXED_DENOMINATOR,0.438024*FIXED_DENOMINATOR,
0.435587*FIXED_DENOMINATOR,0.433146*FIXED_DENOMINATOR,0.430701*FIXED_DENOMINATOR,0.428252*FIXED_DENOMINATOR,0.425799*FIXED_DENOMINATOR,0.423341*FIXED_DENOMINATOR,0.420878*FIXED_DENOMINATOR,0.418411*FIXED_DENOMINATOR,
0.415938*FIXED_DENOMINATOR,0.413460*FIXED_DENOMINATOR,0.410977*FIXED_DENOMINATOR,0.408489*FIXED_DENOMINATOR,0.405994*FIXED_DENOMINATOR,0.403494*FIXED_DENOMINATOR,0.400987*FIXED_DENOMINATOR,0.398474*FIXED_DENOMINATOR,
0.395955*FIXED_DENOMINATOR,0.393428*FIXED_DENOMINATOR,0.390895*FIXED_DENOMINATOR,0.388355*FIXED_DENOMINATOR,0.385807*FIXED_DENOMINATOR,0.383251*FIXED_DENOMINATOR,0.380688*FIXED_DENOMINATOR,0.378116*FIXED_DENOMINATOR,
0.375536*FIXED_DENOMINATOR,0.372947*FIXED_DENOMINATOR,0.370350*FIXED_DENOMINATOR,0.367743*FIXED_DENOMINATOR,0.365127*FIXED_DENOMINATOR,0.362501*FIXED_DENOMINATOR,0.359865*FIXED_DENOMINATOR,0.357218*FIXED_DENOMINATOR,
0.354561*FIXED_DENOMINATOR,0.351893*FIXED_DENOMINATOR,0.349214*FIXED_DENOMINATOR,0.346523*FIXED_DENOMINATOR,0.343821*FIXED_DENOMINATOR,0.341106*FIXED_DENOMINATOR,0.338378*FIXED_DENOMINATOR,0.335637*FIXED_DENOMINATOR,
0.332883*FIXED_DENOMINATOR,0.330115*FIXED_DENOMINATOR,0.327332*FIXED_DENOMINATOR,0.324535*FIXED_DENOMINATOR,0.321722*FIXED_DENOMINATOR,0.318894*FIXED_DENOMINATOR,0.316050*FIXED_DENOMINATOR,0.313189*FIXED_DENOMINATOR,
0.310311*FIXED_DENOMINATOR,0.307415*FIXED_DENOMINATOR,0.304501*FIXED_DENOMINATOR,0.301568*FIXED_DENOMINATOR,0.298615*FIXED_DENOMINATOR,0.295642*FIXED_DENOMINATOR,0.292649*FIXED_DENOMINATOR,0.289633*FIXED_DENOMINATOR,
0.286595*FIXED_DENOMINATOR,0.283534*FIXED_DENOMINATOR,0.280449*FIXED_DENOMINATOR,0.277339*FIXED_DENOMINATOR,0.274204*FIXED_DENOMINATOR,0.271041*FIXED_DENOMINATOR,0.267851*FIXED_DENOMINATOR,0.264632*FIXED_DENOMINATOR,
0.261383*FIXED_DENOMINATOR,0.258103*FIXED_DENOMINATOR,0.254791*FIXED_DENOMINATOR,0.251444*FIXED_DENOMINATOR,0.248063*FIXED_DENOMINATOR,0.244644*FIXED_DENOMINATOR,0.241188*FIXED_DENOMINATOR,0.237691*FIXED_DENOMINATOR,
0.234152*FIXED_DENOMINATOR,0.230569*FIXED_DENOMINATOR,0.226940*FIXED_DENOMINATOR,0.223262*FIXED_DENOMINATOR,0.219534*FIXED_DENOMINATOR,0.215751*FIXED_DENOMINATOR,0.211912*FIXED_DENOMINATOR,0.208014*FIXED_DENOMINATOR,
0.204051*FIXED_DENOMINATOR,0.111123*FIXED_DENOMINATOR,0.108845*FIXED_DENOMINATOR,0.106524*FIXED_DENOMINATOR,0.104159*FIXED_DENOMINATOR,0.101744*FIXED_DENOMINATOR,0.099278*FIXED_DENOMINATOR,0.096756*FIXED_DENOMINATOR,
0.094174*FIXED_DENOMINATOR,0.091526*FIXED_DENOMINATOR,0.088806*FIXED_DENOMINATOR,0.086008*FIXED_DENOMINATOR,0.083124*FIXED_DENOMINATOR,0.080144*FIXED_DENOMINATOR,0.077056*FIXED_DENOMINATOR,0.073849*FIXED_DENOMINATOR
};

void DelayTenthMs(int delay)
{
	int n;
	// this delay takes 0.1ms
	while(delay--)
	{
	for(n = 0; n < 1000; n++) 
	{
	// the following must be around a 2us delay. MIPS=10, so each nop takes 100ns. 
	_asm
		nop
	_endasm
	}	
   }
}

void StartWrite(void)
{
    /*
     * A write command can be prematurely terminated by MCLR or WDT reset
     */
    byte x;

	DelayTenthMs(10);		// protection against too many writes and wearing out the memory!
	/*
	x=TMR1L;
	while(x!=0)
	{
     x=TMR1L;    
    }   	
    */
	INTCONbits.GIE=0;
	EECON2 = 0x55;
    EECON2 = 0xAA;
    EECON1bits.WR = 1;
    INTCONbits.GIE=1;
}
//end StartWrite

byte randomNumber(void)
{
	byte u;

	u=TMR3L+TMR1L+TMR2;
	return u;
}

int ReadEEPROM(byte address)
{
    EECON1 = 0x00;
    EEADR =  address;
    EECON1bits.RD = 1;
    return (int)EEDATA;
}

byte WriteEEPROM(byte address, byte data)
{
		byte i;
		i=ReadEEPROM(address);
		if(i!=data)
		{
		EEADR=(byte)address;
        EEDATA = data;
        EECON1 = 0b00000100;    	// Setup writes: EEPGD=0,WREN=1
        StartWrite();
        while(EECON1bits.WR!=0);       	// Block till WR bit is clear
		}
		return ReadEEPROM(address);
}

//*****************************************************************************
// Keys pressed FIFO functions
//*****************************************************************************
byte getKey(void)
{
	byte g;
	if(keyFull>0)
    	{
	    g=keyBuffer[keyGetPtr];
	    keyGetPtr++;
	    if(keyGetPtr>=KEY_BUFFER_SIZE)keyGetPtr=0;
	    keyFull--;
	    } else g=0xFF;
	return g;
}

void putKey(byte k)
{
    if(keyFull<KEY_BUFFER_SIZE)
     {
	 keyBuffer[keyPtr]=k;
	 keyPtr++;
	 if(keyPtr>=KEY_BUFFER_SIZE)keyPtr=0;
	 keyFull++;
	 }    
}    
/*
byte noKeys(void)
{
	if(keyFull>0)return 0; 
	else return 1;
}
*/

void ackKey(void)
{
	  ir_code=0xFFFF;
	  key=0;
	  ir_rdy=0;
	  ir_timer=0;
	  ir_rpt=0;
}

void initKeys(void)
{
    keyGetPtr=0;
    keyFull=0;   
    keyPtr=0;
}    

void blinkColour(byte times, byte colour)
{
	byte i, c;
	i=0;
	c=colour & 7;
	while(i< times)
	{
	if(c==RED1)RLED1=0;
	else
	if(c==GREEN1)GLED1=0;
	else
	if(c==BLUE1)BLED1=0;
	else
	if(c==RED2)RLED2=0;
	else
	if(c==GREEN2)GLED2=0;
	else BLED2=0;
	DelayTenthMs(MINOR_DELAY);
	PORTA|=LEDSOFFA;
	PORTB|=LEDSOFFB;
	DelayTenthMs(MAJOR_DELAY);
	i++;
	}
}

void blinkString(rom char* instring)
{
	// the input is a string of couplets of the form [ b | B | r | R | g | G ] followed by a single digit N (N=1 to 9).
	unsigned char c, d;
	
	while((*instring)!=0)
	{
		c=*instring++;
		d=*instring++;

		if((d>='1')&&(d<='9'))
		{
			while(d>'0')
			{
			switch(c)
			{
				case 'R':
					RLED1=0;
					break;
				case 'r':
					RLED2=0;
					break;
				case 'B':
					BLED1=0;
					break;
				case 'b':
					BLED2=0;
					break;
				case 'G':
					GLED1=0;
					break;
				case 'g':
					GLED2=0;
					break;
			}
			DelayTenthMs(MINOR_DELAY);
			PORTA|=LEDSOFFA;
			PORTB|=LEDSOFFB;
			DelayTenthMs(BLINK_DELAY);
			d--;
			}		
		}
		DelayTenthMs(MAJOR_DELAY);
	}
}

void blinkN(byte n)
{
	// blink to indicate the digit n where n is between 0 and 9
	byte b, ba;
	if((n>=1)&&(n<=9))
	{
	// Roman Numerals inspired code
	if(n<4)
	{
	b=0;
	ba=0;
	} else
	if(n==4)
	{
	b=0;
	n=1;
	ba=1;
	} else
	if(n<9)
	{
	b=1;
	n-=5;
	ba=0;
	} else
	{
	b=0;
	n=1;
	ba=2;
	}
	blinkColour(b, BLUE2);
	blinkColour(n, GREEN1);
	blinkColour(ba, BLUE2);
	} else 
	if(n==0)
	{
	blinkColour(1, RED1);
	}
	DelayTenthMs(MAJOR_DELAY);
}

void blinkMode(byte m)
{
	DelayTenthMs(MAJOR_DELAY);
	switch(m)
	{
		case NO_MODE:
			blinkString((rom char*)"G1g2");
			break;
		case ZV_MODE:
			blinkString((rom char*)"B1b2");
			break;
		case FLASHING_MODE:
			blinkString((rom char*)"g1G2");
			break;
		case RANDOM_MODE:
			blinkString((rom char*)"b1B2");
			break;
		case BABY_MODE:
			blinkString((rom char*)"r1R2");
			break;
		default:
			blinkString((rom char*)"R3r3");
			break;
	}
}


void resetTimeOut(void)
{
	timeOutCounter=timeOutPeriod;
	seconds=0;
	eternalSeconds=0;
	flashState=0;
}

int divideAByB(int A, int B)
{
	int i;
	// slow and dirty division routine to avoid having to link in the maths library 
	i=0;
	while(A>=B)
	{
	A-=B;
	i++;
	}
	return i;
}

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	byte b;
	byte c;
	
    if(INTCONbits.INT0IE & INTCONbits.INT0IF)
	{
		CCP1CON=0;
		switch(mode)
			{
				case RANDOM_MODE:
				case BABY_MODE:				
				case NO_MODE:
							if(reqLampBrightness>reqQuiescentBrightness)c=reqLampBrightness; else c=reqQuiescentBrightness;
							break;				
				case ZV_MODE:
							if(reqLampBrightness<ZV_THRESHOLD)
							{		
							c=FULLYOFF;
							} else
							{
							c=FULLYON;
							}
							break;

				case FLASHING_MODE:
							if(flashState==0)c=FULLYON; else c=FULLYOFF;
							break;

				default:
							c=FULLYOFF;
							b=0x00;
							break;
			}
		//PIE1bits.CCP1IE=0;
		phaseCounter=TMR1L;
		phaseCounter+=(((int)TMR1H)<<8);
		setBrightness=c;
		if(INTCON2bits.INTEDG0==0)
		{
			if(c>=limiting0)c=limiting0;
			lampBrightness=DimmingCurve[c];
 	        TMR1H=((byte)offset);
			TMR1L=((byte)(offset>>8));
			phaseCounter-=offset;
    		INTCON2bits.INTEDG0=1; 
		}
        else 
        {
			if(c>=limiting1)c=limiting1;
    		lampBrightness=DimmingCurve[c];
	        TMR1H=((byte)offset1);
			TMR1L=((byte)(offset1>>8));
			phaseCounter-=offset1;
            INTCON2bits.INTEDG0=0;
        }   
		CCPR1L=lampBrightness;
		CCPR1H=(lampBrightness>>8);
		PIR1bits.CCP1IF=0;
		if(c==FULLYOFF)b=0x00; else b=0x08;
		CCP1CON=b;							// choose CCP1 mode Compare Mode, Set Output On Match
		PIR1bits.CCP1IF=0;
		PIE1bits.CCP1IE=1;
		INTCONbits.INT0IF=0;
	}

	if(PIE1bits.CCP1IE & PIR1bits.CCP1IF)
	{
		// PIE1bits.TMR2IE=0;
		if(CCP1CON==0x08)
		{ 
		if(lampBrightness>MAX_PHASE_OFFSET)lampBrightness=MAX_PHASE_OFFSET;
		CCPR1L=(lampBrightness+PULSE_PERIOD);
		CCPR1H=((lampBrightness+PULSE_PERIOD)>>8);
		PIR1bits.CCP1IF=0; 
		CCP1CON=0x09;
		} else CCP1CON=0;
		PIR1bits.CCP1IF=0; 
		// PIE1bits.TMR2IE=1;
	}

/*
	if(INTCONbits.RBIE & INTCONbits.RBIF)
	{
		// interrupt on pin change (PORTB) note: this interrupt can be enabled when we go to sleep to wake up on the stimulus from the infra red receiver connected to RB4
		blinkColour(3, RED2);
		b=PORTB;				// must read PORTB to end the mismatch condition!
		INTCONbits.RBIF=0;
	}

*/
}

#pragma interruptlow myISR
void myISR(void)
{
    int x;
	byte y;

	if(PIE1bits.TMR2IE & PIR1bits.TMR2IF)
	{
		if(ir_rdy==0)
		{
    	TRISBbits.TRISB3=1;
        ir_receive();
    	TRISBbits.TRISB3=0;
        }
		else 
		if(ir_rdy==1)
		{		
	 		if((key==0)&&(ir_rpt>=REPEAT_TIMES))
	 		{ 
	 		key=(0x1000 | ir_cmd);       // NB: the 0x1000 is added to force it always to be non-zero.
	 		ir_rdy++;
	 		} else 
	 		{ 
	 		ir_state=0; 
	 		ir_timer=0; 
	 		ir_rdy=0; 
	 		}
		 } 
		 else
		 if(ir_rdy==2)
		 {
         key=translateIRCode(key);
	     ir_rdy++;
	     } else
	     if(ir_rdy==3)
	     {
    	 putKey(key); 
    	 if(mode!=RANDOM_MODE)resetTimeOut(); else timeOutCounter=0;
	     ir_rdy++;
	     }    		
		PIR1bits.TMR2IF=0;
	}	

	if(PIE2bits.TMR3IE & PIR2bits.TMR3IF)
	{
		TMR3H=0x0B;
		TMR3L=0xDC;		// set a 62500 period for 20Hz
		tickCounter++;
		eternalSeconds++;
		if(eternalSeconds>=flashModulus){ if(flashModulus==1)flashState=0; else { flashState++; if(flashState>=4)flashState=0; } eternalSeconds=0; }
				
        if(tickCounter>=20)
		{
		    if(ir_rdy>=4)
		    {
    	    ackKey();
    		}    
    	    eternalSeconds++;
			tickCounter=0;
			seconds++;
			if(seconds>=60)
			{
			seconds=0;
			if(timeOutCounter>0)
				{
				if(mode==BABY_MODE)
				{
				x=divideAByB(targetBrightness, timeOutCounter);
				if(x<=0)x=1;
				targetBrightness-=x;
				}
				timeOutCounter--;				// counts down minutes
				}
			}
		}	
		PIR2bits.TMR3IF=0;
	}

	if(PIE1bits.TMR1IE & PIR1bits.TMR1IF)
	{
		  PIR1bits.TMR1IF=0;
	}
}

/*
byte getBootMode(void)
{
	byte bm;
	bm=1;
	TRISBbits.TRISB6=1;	// make an input
	OUTPUTBOOT=0;
	DelayTenthMs(1);
	if(INPUTBOOT!=0)bm=0;
	OUTPUTBOOT=1;
	DelayTenthMs(1);
	if(INPUTBOOT!=1)bm=0;
	TRISBbits.TRISB6=0; // make output again
	return bm;
}
*/

void initLEDS(void)
{
	CCP1CON=0;
	ADCON1=0x7F;			// configure as digital pins rather than ADC system pins
	PORTA=0xFF;
	PORTB=0xF3;
	LATB=0xF3;
	LATA=0xFF;
	TRISA=0xE0;
	TRISB=0x35;
	// now detect the boot mode
	//bootMode=getBootMode();
}

void initOutput(void)
{
	tickCounter=0;			
	seconds=0;				// counts +1 in one second
	TMR1H=0;
	TMR1L=1;
	T1CON=0xB1;
	IPR1bits.TMR1IP=0;		// interrupt priority
	PIE1bits.TMR1IE=1;		// Timer 1 Interrupt not currently used...
	CCP1CON=0;
#if(LIGHTSYSTEMON==1)
	CCP1CON=0x08;			// choose CCP1 mode Compare Mode, Set Output On Match
#endif	

	INTCONbits.INT0IF=0;	//
	INTCONbits.INT0IE=LIGHTSYSTEMON;	// enable INT0 external interrupt
	DelayTenthMs(MAINSTARTUP_DELAY);

	IPR1bits.CCP1IP=1;		//  interrupt priority for compare
	PIE1bits.CCP1IE=LIGHTSYSTEMON;		// enable compare interrupt
}
	
void initSequence(void)
{
    
	CCP1CON=0;
	PIR1=0;
	PIR2=0;
	PIE1=0;
	PIE2=0;
	INTCON=0;

	RCONbits.IPEN=1;	// enable prioritized interrupts!
	mode=NO_MODE;
	flashModulus=FLASH_MODULUS;
	defineDefaultRemoteControl();
	reqLampBrightness=0;
	reqQuiescentBrightness=0;
	initOutput();
	timeOutPeriod=DEFAULT_TIMEOUT_PERIOD;
	resetTimeOut();
	ackKey();
	initLEDS();
	initKeys();
	initRC5();
	// now setup up the timer for the IR decoding
	T3CON=0x31;				// set up Timer 3 to count Fosc/4 pulses (10MHz) with 1:8 prescaler
							// since Timer 3 is 16 bits the interrupt occurs at 20Hz roughly (note that we are setting a period of 62500=0xF424=-0x0BDC
	IPR2bits.TMR3IP=0;
	PIE2bits.TMR3IE=1;
	IPR1bits.TMR2IP=IRSERVICEHIGH;		// priority
	PIE1bits.TMR2IE=IRSYSTEMON;
	TMR2=0;
	PR2=40;
	T2CON=0x06;		    // select 1:1 post scale and 1:16 prescale of Fosc/4=10 MHz for a timeout frequency around 1/64us
	INTCONbits.GIEL=1;	// enable low interrupts!
	INTCONbits.GIE=1;	// enable all interrupts!
}

/*
void initSequence(void)
{
	key=0;
	initLEDS();
	initRC5();
	// now setup up the timer for the IR decoding
	PIR1bits.TMR2IF=0;
	IPR1bits.TMR2IP=1;	// high priority
	PIE1bits.TMR2IE=1;
	TMR2=0;
	PR2=80;
	T2CON=0x05;		// select 1:1 post scale and 1:4 prescale of Fosc/4=5 MHz for a timeout frequency around 1/64us
	//
	RCONbits.IPEN=1;	// enable prioritized interrupts!
	INTCONbits.GIEL=0;
	INTCONbits.GIE=1;	// enable interrupts!
}
*/

/*
int setLampBrightness(int level)
{
	//
	// range goes from 0 (fully on) to about (4095) fully off
	// ie ON_BRIGHTNESS to OFF_BRIGHTNESS
	//
	reqlampBrightness=level+OFFSET_PERIOD;
	return 0;
}	
*/

/*
void setLampBrightnessLevel(byte level)
{
	reqLampBrightness=level;
}	

void setQuiescentBrightnessLevel(byte level)
{
	reqQuiescentBrightness=level;
}	
*/

/*
void enterPowerSaving(void)
{
	byte b;
	// turn off some peripherals not needed when lamp is 'off'
	INTCON2bits.RBIP=1;	// priority
	b=PORTB;			// clears any pending change on port b
	INTCONbits.RBIF=0;
	INTCONbits.RBIE=1;	// enable interrupt on change pin (RB4 is input of data from infra red receive module)
	OSCCON=0;
}
*/

/*
void exitPowerSaving(void)
{
	// undo what enterPowerSaving() does
	OSCCON=0;
	INTCONbits.RBIE=0;	// disable the interrupt on change pin 
	INTCONbits.RBIF=0;
}

*/

/*
Note: doubles don't work!
int setLampBrightnessPercent(int percent)
{
	// where argument is a percentage from 0 to 100
	double f;
	
	if(percent<=0)percent=0; else if(percent>=100)percent=100;
	f=(double)percent;
	f=f/100.0;
	f=OFF_BRIGHTNESS*(1.0-f);
	percent=(int)f;
	setLampBrightness(percent);
	return 0;
}
*/	

void blinkKey(byte keycode, byte argtarget)
{
	switch(keycode)
	{
		case KEYZERO:
		case KEYONE:
		case KEYTWO:
		case KEYTHREE:
		case KEYFOUR:
		case KEYFIVE:
		case KEYSIX:
		case KEYSEVEN:
		case KEYEIGHT:
		case KEYNINE:
			blinkN(keycode-1);
			break;
		case KEYDIMUP:
			blinkString((rom char*)"g1");
			break;
		case KEYDIMDOWN:
			blinkString((rom char*)"r1");
			break;
		case KEYMODEUP:
			blinkString((rom char*)"b1B1");
			break;
		case KEYMODEDOWN:
			blinkString((rom char*)"B1b1");
			break;
		case KEYTOGGLE:
			if(argtarget!=0)
			{
			blinkString((rom char*)"B1b2");
			}
			else 
			{
			blinkString((rom char*)"b1B2");
			}
			break;	
		case KEYOK:
			blinkString((rom char*)"b2");
			break;
		case KEYMENU:
			blinkString((rom char*)"g1R1");
			break;
		case KEYINFO:
			blinkString((rom char*)"g1B1");
			break;
		case KEYPLAY:
			blinkString((rom char*)"g2");
			break;
		case KEYRECORD:
			blinkString((rom char*)"r2");
			break;
		default:
			if(badTimes<5)blinkString((rom char*)"R3");
			else blinkString((rom char*)"r3");
			break;
	}
}

rom char enterMenuString[]="b1B1g1G1r1R1";
rom char exitMenuString[]="R1r1G1g1B1b1";

int enterNumber(void)
{
	byte i, myKey, sign;
	int x;
	//
	sign=0;
	x=0;
	blinkString(enterMenuString);
	i=0;
	while(i==0)
	{
     initKeys();
     ackKey();
	 while(keyFull==0);
	 myKey=getKey();
	 switch(myKey)
		 {
		 case KEYZERO:
		 case KEYONE:
		 case KEYTWO:
		 case KEYTHREE:
		 case KEYFOUR:
		 case KEYFIVE:
		 case KEYSIX:
		 case KEYSEVEN:
		 case KEYEIGHT:
		 case KEYNINE:
				myKey--;
				break;
		 case KEYOK:
				i=1;
				break;

		 case KEYTOGGLE:
				sign^=1;
				break;

		 case KEYDIMUP:
				sign=0;
				break;

		 case KEYDIMDOWN:
				sign=1;
				break;

		 default:
				myKey=0;
				break;
		}
		if(i==0)
		{ 
		 DelayTenthMs(MAJOR_DELAY);
		 if((myKey!=KEYTOGGLE)&&(myKey!=KEYDIMUP)&&(key!=KEYDIMDOWN))
		 {
		 blinkN(myKey); 
		 while(x>999)
			{ 
			x-=1000; 
			}
	 	 x=x+x+x+x+x;
	 	 x=x+x;
	 	 x+=myKey;
		 } else
		 blinkKey(KEYTOGGLE, sign);
		}
	 }
  	 blinkString(exitMenuString);
	 if(sign!=0)
	 {
	 	x=~x;
		x++;		// x=-x
	 }
	 return x;
}

rom int PowersOfTen[5]={ 10000, 1000, 100, 10, 1 };

void blinkNumber(int x, byte sign)
{
	byte i, j, blanking;
	int y;
    //
	DelayTenthMs(MAJOR_DELAY);
	i=0;
	blanking=1;
	if(x!=0)
	{
	if(x<0)
	{
	blinkKey(KEYTOGGLE, 1);
	x=~x;
	x++;			
	// x=-x
	} else
	{
	if(sign)blinkKey(KEYTOGGLE, 0);
	}
	 while(i<5)
	 {
		y=PowersOfTen[i];
		j=0;
		while(x>=y)
		{
		x-=y;
		j++;
		}
		if(j!=0)blanking=0;
		if(blanking==0)blinkN(j);
		DelayTenthMs(BLINKDIGIT_DELAY);
		i++;
	 }
	} 
	else 
	{ 
	blinkN(0); 
	}
}

void enterQuiescentLevel(void)
{
	int x; 
	byte m, i, myKey;
	int brightness;
	
	blinkString(enterMenuString);
	m=mode;
	reqQuiescentBrightness=0;
	mode=NO_MODE;
	brightness=0;
	i=0;
	while(i==0)
	{
	 if(brightness<0)brightness=0; else if(brightness>255)brightness=255;
     reqLampBrightness=brightness;
     initKeys();
     ackKey();
     while(keyFull==0);
     myKey=getKey();
	 switch(myKey)
		 {
		 case KEYZERO:
		 case KEYONE:
		 case KEYTWO:
		 case KEYTHREE:
		 case KEYFOUR:
		 case KEYFIVE:
		 case KEYSIX:
		 case KEYSEVEN:
		 case KEYEIGHT:
		 case KEYNINE:
				x=((int)myKey)-1;
				brightness=(int)(((int)x)<<3);			// *8				
				blinkN(x);
				break;
		 case KEYOK:
				i=1;
				break;
		 case KEYDIMUP:
				brightness+=4;
				blinkKey(myKey, 0);
				break;
		 case KEYDIMDOWN:
				brightness-=4;
				blinkKey(myKey, 0);
				break;

		 default:
				break;
		}
	}
	reqQuiescentBrightness=(byte)brightness;
	reqLampBrightness=0;
	WriteEEPROM(QUIESCENT_ADR, (byte)brightness);
	blinkString(exitMenuString);
	mode=m;
}

void doMenu(byte menu)
{
	switch(menu)
	{
		case MENU_QUIESCENT:
			enterQuiescentLevel();
			break;
		case MENU_TIMEOUTPERIOD:
			timeOutPeriod=enterNumber();
			WriteEEPROM(TIMEOUTHI_ADR, (timeOutPeriod>>8));
			WriteEEPROM(TIMEOUTLO_ADR, timeOutPeriod);
			break;
		case MENU_FLASHMODULUS:
			flashModulus=enterNumber();
			if(flashModulus>254)flashModulus=255;
			WriteEEPROM(FLASHMODULUS_ADR, flashModulus);
			break;

		case MENU_ADDRESS:
			address=enterNumber();
			if(address>9)address=0;
			WriteEEPROM(ADDRESS_ADR, address);
			break;
		case MENU_LIMITING0:
			limiting0=enterNumber();
			WriteEEPROM(LIMITING0_ADR, limiting0);
			break;
		case MENU_LIMITING1:
			limiting1=enterNumber();
			WriteEEPROM(LIMITING1_ADR, limiting1);
			break;
		case MENU_OFFSET:
			offset=enterNumber();
			WriteEEPROM(OFFSETLO_ADR, offset);
			WriteEEPROM(OFFSETHI_ADR, (offset>>8));
			break;
			
		case MENU_OFFSET1:
			offset1=enterNumber();
			WriteEEPROM(OFFSET1LO_ADR, offset1);
			WriteEEPROM(OFFSET1HI_ADR, (offset1>>8));
			break;

		case MENU_LOOPDELAY:
		    loopDelay=enterNumber();
		    WriteEEPROM(LOOPDELAY_ADR, loopDelay);
		    break;
		    	
		case MENU_RESET:
			WriteEEPROM(REMOTE_CONTROL_DEFS_ADDRESS,0xFF);
			INTCONbits.GIE=0;
			_asm
				reset
			_endasm
		default:
			break;
	}
	initKeys();
	ackKey();
}

void doInfo(byte info)
{
	switch(info)
	{
		case INFO_TIMEOUTPERIOD:
			blinkNumber(timeOutPeriod, 1);
			break;
		case INFO_VERSION:
			blinkNumber(VERSION, 1);
			break;
		case INFO_MAINSFREQ:
			// divide the phase counter by 25 to get a three digit mains freq with the last digit being the tenths of a Hertz. eg. 504 indicates 50.4Hz
			blinkNumber(divideAByB(phaseCounter,25), 1);
			break;
       	case INFO_ADDRESS:
			blinkNumber(address & 0x0F, 1);
			break;
		case INFO_LIMITING0:
			blinkNumber(limiting0, 1);
			break;
		case INFO_LIMITING1:
			blinkNumber(limiting1, 1);
			break;
		case INFO_OFFSET:
			blinkNumber(offset, 1);
			break;
		case INFO_OFFSET1:
			blinkNumber(offset1, 1);
			break;
        case INFO_LOOPDELAY:
            blinkNumber(loopDelay, 1);
            break;
       	case INFO_FLASHMODULUS:
			blinkNumber(flashModulus, 1);
			break;
		default:
			break;
	}
	initKeys();
	ackKey();
}

void ackNotAddressed(void)
{
	DelayTenthMs(MAJOR_DELAY);
	blinkString((rom char*)"r2");
}

byte isAddressed(void)
{
	if((address==0)||((address & 0x0F)==receivedAddress))return 1; else return 0;
}

void main(void)
{
	byte x;
	byte myKey;
	byte previousKey;
	int n;
	//
	setBrightness=0;
	limiting0=0;
	limiting1=0;
	offset=0;
	offset1=0;
	initSequence();
	limiting0=ReadEEPROM(LIMITING0_ADR);
	limiting1=ReadEEPROM(LIMITING1_ADR);
	offset=(ReadEEPROM(OFFSETHI_ADR)<<8)+(ReadEEPROM(OFFSETLO_ADR));
	offset1=(ReadEEPROM(OFFSET1HI_ADR)<<8)+(ReadEEPROM(OFFSET1LO_ADR));
	mode=ReadEEPROM(MODE_ADR);
	flashModulus=ReadEEPROM(FLASHMODULUS_ADR);
	timeOutPeriod=(ReadEEPROM(TIMEOUTHI_ADR)<<8)+(ReadEEPROM(TIMEOUTLO_ADR));
	address=ReadEEPROM(ADDRESS_ADR);
	loopDelay=ReadEEPROM(LOOPDELAY_ADR);
	reqQuiescentBrightness=ReadEEPROM(QUIESCENT_ADR);
	targetBrightness=ReadEEPROM(LAMPBRIGHTNESS_ADR);
    key=0;
	previousKey=0;
	targetACK=0;
	badCode=0;
	badTimes=0;
	receivedAddress=-1;
	resetTimeOut();
	while(1)
	{
	 if(keyFull>0)
	 {
		if(badTimes>=10)
		{
		// setLampBrightnessLevel(0);
		reqLampBrightness=0;
		x=mode;
		mode=NO_MODE;
		reDefineRemoteControlCodes();
		mode=x;
		badCode=0;
		badTimes=0;
		}
		
		myKey=getKey();
		if((myKey & 0x80)==0)
		{
		badCode=0;
		badTimes=0;
 		}

        blinkKey(myKey, targetBrightness); 
	    switch(myKey)
		 {
			 case KEYZERO:
			 case KEYONE:
			 case KEYTWO:
			 case KEYTHREE:
			 case KEYFOUR:
			 case KEYFIVE:
			 case KEYSIX:
			 case KEYSEVEN:
			 case KEYEIGHT:
			 case KEYNINE:
				x=myKey-1;
				if(previousKey==KEYOK)
				{
					receivedAddress=x;
				} else
				if(previousKey==KEYINFO)
				{
					doInfo(x);
					myKey=0;
				} else
				if(isAddressed())
				{
					if(previousKey==KEYMENU)
					{
					// do the menu!
					doMenu(x);
					myKey=0;
					} else
					if(mode!=FLASHING_MODE)
					{
					targetBrightness=(x<<4)-x-x;			// *14
					targetBrightness=(targetBrightness<<1); // *2=*28
					}
				} else ackNotAddressed();
				break;

			 case KEYMODEUP:
				if(isAddressed())
				{
				if(mode<(ALL_MODES-1))mode++;
				WriteEEPROM(MODE_ADR, mode);
				blinkMode(mode);
				} else ackNotAddressed();
				break;
			 case KEYMODEDOWN:
				if(isAddressed())
				{
				if(mode>0)mode--;
				WriteEEPROM(MODE_ADR, mode);
				blinkMode(mode);
				} else ackNotAddressed();
				break;

			 case KEYTOGGLE:
				if(isAddressed())
				{
				  if(targetBrightness!=0)
		 		  {
			  		targetBrightness=0; 
		 		  }
	 			  else 
				  {
			 		targetBrightness=DIMMING_LEVELS-1; 
				  }
				} else ackNotAddressed();
				break;	

        	 case KEYDIMUP:
					if(isAddressed())
					{
					if((mode==NO_MODE)||(mode==BABY_MODE))targetBrightness+=10; else if(mode==ZV_MODE)targetBrightness=FULLYON;
					} else ackNotAddressed();
				break;

			 case KEYDIMDOWN:
					if(isAddressed())
					{
					if((mode==NO_MODE)||(mode==BABY_MODE))targetBrightness-=10; else if(mode==ZV_MODE)targetBrightness=0;
					} else ackNotAddressed();
				break;
   
			 case KEYOK:
				doInfo(INFO_ADDRESS);
				DelayTenthMs(MAJOR_DELAY);
				break;

			 case KEYMENU:
				break;

			 case KEYINFO:
				break;

			 case KEYPLAY:
				if(isAddressed())
				{
				mode=ReadEEPROM(SAVEMODE_ADR);
				WriteEEPROM(MODE_ADR, mode);
				targetBrightness=ReadEEPROM(SAVEBRIGHTNESS_ADR);
				targetACK=0;
				} else ackNotAddressed();
				break;

			case KEYRECORD:
				if(isAddressed())
				{
				WriteEEPROM(SAVEMODE_ADR, mode);
				WriteEEPROM(SAVEBRIGHTNESS_ADR, setBrightness);
				} else ackNotAddressed();
				break;

			default:
				if(badTimes==0)
				{
				badCode=myKey;
				badTimes++;
				} else
				{
    			if(badCode==myKey)badTimes++; else badTimes=0;
				}
				break;
		}
		previousKey=myKey;
		if(mode==FLASHING_MODE)eternalSeconds=(flashModulus-2);
	}

    if((timeOutPeriod>0)&&(mode!=FLASHING_MODE)&&(mode!=RANDOM_MODE))
	{ 
        if(timeOutCounter<=0)targetBrightness=0; 
        else 
        {
         if((seconds>30)&&((seconds & 3)==0))
            {
                if(((mode==NO_MODE)&&(timeOutCounter==1))||(mode==BABY_MODE))blinkString((rom char*)"r1R1");
            }    
        }    
	} else
	if(mode==RANDOM_MODE)
	{
		if(timeOutCounter<=0)
				{ 
				if(targetBrightness!=0)targetBrightness=0; 
				else targetBrightness=FULLYON;
				// now seed the timeOutCounter with a new "random" value in minutes...
				// between 5 and 127+5 minutes, say
				x=(randomNumber() & 0x7F)+5;
				timeOutCounter=(int)x;
				blinkNumber(timeOutCounter, 1);
				}
	}

	if(targetBrightness>=DIMMING_LEVELS)targetBrightness=DIMMING_LEVELS-1;
	else if(targetBrightness<0)targetBrightness=0;
	//
	// Target convergence:
	//
	if(mode==FLASHING_MODE)
	{
		if(targetACK==0)
			{
			targetACK=1;
			blinkMode(mode);
			}

	} else
	if((mode==RANDOM_MODE)||(mode==ZV_MODE))
	{
    	// Digital convergence
    	if((setBrightness==0)^(targetBrightness==0))
    	{
    	    if(targetBrightness==0)reqLampBrightness=0;			//setLampBrightnessLevel(0);
    	    else
    	    { 
        	//setLampBrightnessLevel(FULLYON);
			reqLampBrightness=FULLYON;
        	targetBrightness=FULLYON;
            }   	
    	targetACK=0;
   		DelayTenthMs(loopDelay);
    	}
    	
    } else
	if((mode==NO_MODE)||(mode==BABY_MODE))
    {   
        // Continuous convergence	
	       if(setBrightness<targetBrightness)
	       {
            targetACK=0;
    	    //setLampBrightnessLevel(setBrightness+1);
			reqLampBrightness=setBrightness+1;
       		DelayTenthMs(loopDelay);
           }       
           else
    	   if(setBrightness>targetBrightness)
	       {
    	   targetACK=0;
    	   //setLampBrightnessLevel(setBrightness-1);
		   reqLampBrightness=setBrightness-1;
     	   DelayTenthMs(loopDelay);
           }   	
	 }

	if((setBrightness==targetBrightness)&&(targetACK==0)&&(keyFull==0))
	{
		targetACK=1;
    	WriteEEPROM(LAMPBRIGHTNESS_ADR, targetBrightness);
		if((previousKey!=KEYDIMUP)&&(previousKey!=KEYDIMDOWN))blinkMode(mode);
	}   

	if((previousKey==KEYMENU)||(previousKey==KEYINFO))
	{
	blinkKey(previousKey,0);
	DelayTenthMs(MAJOR_DELAY);
	} else
	if(previousKey==KEYOK)
	{
	blinkKey(previousKey,0);
	doInfo(INFO_ADDRESS);
	DelayTenthMs(MAJOR_DELAY);
	}
  }
}
